<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\UploadedFile;
use Exception;
use Illuminate\Support\Facades\Log;

class WhatsAppApiService
{
    protected PendingRequest $client;
    protected string $baseUrl; // Add this property
    protected int $timeout; // Add this property

    public function __construct()
    {
        $this->baseUrl = config('services.whatsapp.api_url'); // Store in property
        $this->timeout = config('services.whatsapp.timeout', 30);

        if (!$this->baseUrl) {
            throw new Exception('WhatsApp API base URL is not configured.');
        }

        $this->client = Http::baseUrl($this->baseUrl)
            ->timeout($this->timeout) // Set timeout here
            ->acceptJson();
    }

    /**
     * Check connection health status
     */
    public function getHealthStatus(): array
    {
        try {
            // Use the stored client instead of creating new one
            $response = $this->client->get("/health");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json(),
                    'status' => $response->json('status') ?? 'unknown'
                ];
            }

            return [
                'success' => false,
                'status' => 'disconnected',
                'message' => 'API unavailable'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp API Health Check Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'status' => 'error',
                'message' => 'Cannot reach WhatsApp API'
            ];
        }
    }

    /**
     * Get QR code for authentication
     */
    public function getQrCode(): array
    {
        try {
            // Use the stored client instead of creating new one
            $response = $this->client->get("/auth/qr");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'qr' => $response->json('qr'),
                    'authenticated' => $response->json('authenticated', false)
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to fetch QR code'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp QR Fetch Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot generate QR code'
            ];
        }
    }

    /**
     * Send a message
     */
    public function sendMessage(string $to, string $message): array
    {
        try {
            $response = $this->client->post("/send", [
                'to' => $to,
                'message' => $message
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => $response->json('error') ?? 'Failed to send message'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Send Message Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Message delivery failed'
            ];
        }
    }

    /**
     * Reply to a message
     */
    public function replyMessage(string $messageId, string $to, string $message): array
    {
        try {
            $response = $this->client->post("/reply", [
                'messageId' => $messageId,
                'to' => $to,
                'message' => $message
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to send reply'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Reply Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Reply delivery failed'
            ];
        }
    }

    /**
     * Send a file
     */
    public function sendFile(string $to, $file, ?string $caption = null): array
    {
        try {
            $response = $this->client->attach('file', 
                file_get_contents($file->getRealPath()), 
                $file->getClientOriginalName()
            )->post("/send-file", [
                'to' => $to,
                'caption' => $caption
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'File upload failed'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Send File Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot send file'
            ];
        }
    }

    /**
     * Get all contacts
     */
    public function getContacts(): array
    {
        try {
            $response = $this->client->get("/contacts");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'contacts' => $response->json('contacts', [])
                ];
            }

            return [
                'success' => false,
                'contacts' => [],
                'message' => 'Failed to fetch contacts'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Contacts Fetch Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'contacts' => [],
                'message' => 'Cannot retrieve contacts'
            ];
        }
    }

    /**
     * Get contact details
     */
    public function getContact(string $id): array
    {
        try {
            $response = $this->client->get("/contact/{$id}");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'contact' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Contact not found'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Contact Fetch Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot retrieve contact'
            ];
        }
    }

    /**
     * Get all chats
     */
    public function getChats(): array
    {
        try {
            $response = $this->client->get("/chats");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'chats' => $response->json('chats', [])
                ];
            }

            return [
                'success' => false,
                'chats' => [],
                'message' => 'Failed to fetch chats'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Chats Fetch Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'chats' => [],
                'message' => 'Cannot retrieve chats'
            ];
        }
    }

    /**
     * Get all groups
     */
    public function getGroups(): array
    {
        try {
            $response = $this->client->get("/groups");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'groups' => $response->json('groups', [])
                ];
            }

            return [
                'success' => false,
                'groups' => [],
                'message' => 'Failed to fetch groups'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Groups Fetch Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'groups' => [],
                'message' => 'Cannot retrieve groups'
            ];
        }
    }

    /**
     * Get group metadata
     */
    public function getGroupMeta(string $id): array
    {
        try {
            $response = $this->client->get("/group/{$id}/meta");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'group' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Group not found'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Group Meta Fetch Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot retrieve group details'
            ];
        }
    }

    /**
     * Broadcast message to group
     */
    public function broadcastToGroup(string $groupId, string $message): array
    {
        try {
            $response = $this->client->post("/group/broadcast", [
                'groupId' => $groupId,
                'message' => $message
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Broadcast failed'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Group Broadcast Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot send broadcast'
            ];
        }
    }

    /**
     * Set webhook URL
     */
    public function setWebhook(string $url): array
    {
        try {
            $response = $this->client->post("/webhook/set", [
                'url' => $url
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to set webhook'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Webhook Set Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot configure webhook'
            ];
        }
    }

    /**
     * Get current webhook URL
     */
    public function getWebhook(): array
    {
        try {
            $response = $this->client->get("/webhook/get");

            if ($response->successful()) {
                return [
                    'success' => true,
                    'webhook' => $response->json('url')
                ];
            }

            return [
                'success' => false,
                'webhook' => null,
                'message' => 'No webhook configured'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Webhook Get Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'webhook' => null,
                'message' => 'Cannot retrieve webhook'
            ];
        }
    }

    /**
     * Toggle message listening
     */
    public function toggleListen(bool $enable): array
    {
        try {
            $response = $this->client->get("/listen", [
                'enable' => $enable
            ]);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Failed to toggle listening'
            ];
        } catch (Exception $e) {
            Log::error('WhatsApp Listen Toggle Failed', ['error' => $e->getMessage()]);
            return [
                'success' => false,
                'message' => 'Cannot change listening state'
            ];
        }
    }
}