<?php

namespace App\Http\Controllers;

use App\Services\WhatsAppApiService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\View\View;
use Exception;
use Illuminate\Support\Facades\Log;

class MessageController extends Controller
{
    protected WhatsAppApiService $whatsappApi;

    public function __construct(WhatsAppApiService $whatsappApi)
    {
       $this->whatsappApi = $whatsappApi;
    }

    /**
     * Show the form for sending a new message.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $chats = $this->whatsappApi->getChats();
        return view('messages.index', ['chats' => $chats['chats'] ?? []]);
    }

    public function create()
    {
        $contacts = $this->whatsappApi->getContacts();
        return view('messages.send', ['contacts' => $contacts['contacts'] ?? []]);
    }

    public function send(Request $request)
    {
        $validated = $request->validate([
            'to' => 'required|string',
            'message' => 'required|string|max:4096'
        ]);

        $result = $this->whatsappApi->sendMessage($validated['to'], $validated['message']);

        if ($result['success']) {
            return redirect()->route('messages.index')
                ->with('success', 'Message sent successfully');
        }

        return back()->withErrors(['error' => $result['message']]);
    }

    public function reply(Request $request)
    {
        $validated = $request->validate([
            'message_id' => 'required|string',
            'to' => 'required|string',
            'message' => 'required|string|max:4096'
        ]);

        $result = $this->whatsappApi->replyMessage(
            $validated['message_id'],
            $validated['to'],
            $validated['message']
        );

        return response()->json($result);
    }

    public function sendFile(Request $request)
    {
        $validated = $request->validate([
            'to' => 'required|string',
            'file' => 'required|file|max:20480',
            'caption' => 'nullable|string|max:1024'
        ]);

        $result = $this->whatsappApi->sendFile(
            $validated['to'],
            $request->file('file'),
            $validated['caption'] ?? null
        );

        if ($result['success']) {
            return redirect()->route('messages.index')
                ->with('success', 'File sent successfully');
        }

        return back()->withErrors(['error' => $result['message']]);
    }
}