<?php

namespace App\Http\Controllers;

use App\Services\WhatsAppApiService;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Exception;
use Illuminate\Support\Facades\Log;

class GroupController extends Controller
{
    protected WhatsAppApiService $whatsappApi;

    public function __construct(WhatsAppApiService $whatsappApi)
    {
       $this->whatsappApi = $whatsappApi;
    }

    /**
     * Display a list of groups.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $groups = $this->whatsappApi->getGroups();
        $search = $request->query('search');

        $filteredGroups = $groups['groups'] ?? [];

        if ($search) {
            $filteredGroups = array_filter($filteredGroups, function ($group) use ($search) {
                return stripos($group['name'] ?? '', $search) !== false;
            });
        }

        return view('groups.index', [
            'groups' => $filteredGroups,
            'search' => $search
        ]);
    }

    public function show(string $id)
    {
        $group = $this->whatsappApi->getGroupMeta($id);
        
        if (!$group['success']) {
            return redirect()->route('groups.index')
                ->withErrors(['error' => 'Group not found']);
        }

        return view('groups.show', ['group' => $group['group']]);
    }

    public function showBroadcast()
    {
        $groups = $this->whatsappApi->getGroups();
        return view('groups.broadcast', ['groups' => $groups['groups'] ?? []]);
    }

    public function broadcast(Request $request)
    {
        $validated = $request->validate([
            'group_id' => 'required|string',
            'message' => 'required|string|max:4096'
        ]);

        $result = $this->whatsappApi->broadcastToGroup(
            $validated['group_id'],
            $validated['message']
        );

        if ($result['success']) {
            return redirect()->route('groups.index')
                ->with('success', 'Broadcast sent successfully');
        }

        return back()->withErrors(['error' => $result['message']]);
    }
}